;+
; Procedure: tplot_force_monotonic
;
; Purpose:
;    This routine checks tplot variables for sample time (abscissa: data.x) monotonicity
;     and forces them so through removal or replacement of non-monotonic segments, if requested;
;     the corresponding elements of data.y and data.v,v1,v2 are also repaired.
;    Indices of consecutively repeated header times and piece-wise monotonic segment "negative jumps"
;     are identified and used to determine monotonicity of tplot variables.  
;    Three repair methods are available: /first, /second and /sort (see Keywords) 
;    
; Inputs:   [optional] tplot variable name(s) string/array or wild-card name string or tplot variable number(s); same as input for tnames()
;                    
; Keywords:
;           first:        repair method which keeps the 'first' (i.e. older time segment) of the two over-lapping time segments
;           second:       repair method which keeps the 'second' (i.e. newer time segment) of the two over-lapping time segments
;           sort:         repair method which sorts time-lines chronologically, using bsort()
;           
;           keep_repeats: [optional] do not remove consecutively repeated header times
;           force_NaN:    [optional] instead of removing 'bad' elements, replace them with NaN (only applied to data.y and data.v elements, data.x is not modified)
;           
;              
; Outputs:
;           If checking, PASS/FAIL console message per tplot variable
;           If repairing, PASS/FAIL console message per tplot variable and repaired tplot variable(s) (via store_data) 
;              
; Examples:
;           tplot_force_monotonic                           ; check all tplot variables
;           tplot_force_monotonic,'*'                       ; check all tplot variables
;           tplot_force_monotonic,'var1'                    ; check tplot variable named var1
;           tplot_force_monotonic,['var1','var2']           ; check tplot variables named var1 and var2
;           tplot_force_monotonic,[6,7,9]                   ; check tplot variables 6, 7 and 9
;           tplot_force_monotonic,/first                    ; repair all tplot variables using 'first' method, discarding repeated sample times
;           tplot_force_monotonic,/second,/keep_repeats     ; repair all tplot variables using 'second' method, keeping repeated sample times
;           tplot_force_monotonic,/sort                     ; repair all tplot variables using 'sort' method, discarding consecutively repeated sample times
;           tplot_force_monotonic,'var?',/first,/force_nan  ; repair tplot variable(s) in var? using 'first' method and replace bad elements with NaNs
;        
; Notes:
;     1. tplot variables repaired with the /force_nan keyword will not pass a monotonicity check.
;     2. A warning is issued if more than 10% of tplot variable elements are removed or replaced.
;      
; ToDo: nothing yet
;
; $LastChangedBy: $
; $LastChangedDate: $
; $LastChangedRevision: $
; $URL: $
;-

function keep_overlap, i_keep, i_new, time_array, SECOND=SECOND, KEEP_REPEATS=KEEP_REPEATS
;;; function which returns the indices of overlapping time segments to keep
if keyword_set(SECOND) then begin
  if keyword_set(KEEP_REPEATS) then begin i_keep_new = where( time_array[i_new] le min(time_array[i_keep]) )
  endif else i_keep_new = where( time_array[i_new] lt min(time_array[i_keep]) )
endif else begin
  if keyword_set(KEEP_REPEATS) then begin i_keep_new = where( time_array[i_new] ge max(time_array[i_keep]) )
  endif else i_keep_new = where( time_array[i_new] gt max(time_array[i_keep]) )
endelse

if max(i_keep_new) ge 0 then i_keep = [i_keep, i_new[i_keep_new]]

return, i_keep
end

pro tplot_force_monotonic, tplot_vars, FIRST=FIRST, SECOND=SECOND, SORT=SORT, KEEP_REPEATS=KEEP_REPEATS, FORCE_NAN=FORCE_NAN

;;; Get and check tplot variable names/numbers
tplot_vars = tnames(tplot_vars)
if strlen(strjoin(tplot_vars)) eq 0 then begin
  message, 'Invalid tplot variable name(s) or number(s)',/continue
  return
endif
max_var_len = 32

;;; Loop over requested tplot variables
for j=0L,n_elements(tplot_vars)-1L do begin
  tplot_var = tplot_vars[j]
  get_data,tplot_var,data=data,dlimits=dlimits,limits=limits
  time_array = data.x

  ;;; Check if time array has only one element
  if (n_elements(time_array) eq 1) then begin
    dprint,' ',tplot_var,':'+strjoin(strarr(max_var_len-strlen(tplot_var))+' ')+'PASS, variable has one time element'
    continue
  endif
  
  ;;; Identify consecutive repeats and boundaries of piece-wise monotonic segments "negative jumps"
  i_repeat = where( (time_array - shift(time_array,1)) eq 0d)
  i_non_monotonic = where( (time_array - shift(time_array,1)) lt 0d)

  ;;; If time_array is monotonic (i.e. has no negative jumps) and has no repeats, then it PASSES
  if (n_elements(i_non_monotonic) eq 1) and (min(i_repeat) eq -1) then begin 
    dprint,' ',tplot_var,':'+strjoin(strarr(max_var_len-strlen(tplot_var))+' ')+'PASS'
    continue
  endif else begin
    
    ;;; case: tplot variable is non-monotonic
    if keyword_set(FIRST) or keyword_set(SECOND) or keyword_set(SORT) then begin
      if n_elements(i_non_monotonic) eq 1 then begin
        GOTO, ONLY_REPEATS
      endif else i_non_monotonic = i_non_monotonic[1:*]
      
      ;;; Keep first portion of overlaps
      if keyword_set(FIRST) then begin
        i_keep = lindgen(i_non_monotonic[0])
        for k = 0L, n_elements(i_non_monotonic)-2L do begin
            i_new = lindgen(i_non_monotonic[k+1] - i_non_monotonic[k]) + i_non_monotonic[k]
            i_keep = keep_overlap(i_keep,i_new,time_array,KEEP_REPEATS=KEEP_REPEATS)
        endfor
        i_new = lindgen( n_elements(time_array) - i_non_monotonic[k] ) + i_non_monotonic[k]
        i_keep = keep_overlap(i_keep,i_new,time_array,KEEP_REPEATS=KEEP_REPEATS)
      endif 
    
      ;;; Keep second portion of overlaps
      if keyword_set(SECOND) then begin
        n = n_elements(i_non_monotonic)-1
        i_keep = lindgen( n_elements(time_array) - i_non_monotonic[n]) +i_non_monotonic[n]
        for k = n_elements(i_non_monotonic)-2L,0,-1 do begin
            i_new = lindgen(i_non_monotonic[k+1]-i_non_monotonic[k]) + i_non_monotonic[k]
            i_keep = keep_overlap(i_keep,i_new,time_array,/SECOND,KEEP_REPEATS=KEEP_REPEATS)
        endfor
        i_new = lindgen(i_non_monotonic[0])
        i_keep = keep_overlap(i_keep,i_new,time_array,/SECOND,KEEP_REPEATS=KEEP_REPEATS)
      endif
      
      ;;; Sort time-series using bsort()
      if keyword_set(SORT) then i_keep = bsort(time_array)
        
      ;;; Keep repeats or not
      ONLY_REPEATS: if min(i_non_monotonic) le 0 then i_keep = lindgen(n_elements(time_array))
      if not keyword_set(KEEP_REPEATS) then begin
        i_non_repeats = ssl_set_complement([i_repeat],[i_keep])
        if min(i_non_repeats) ne -1 then i_keep = i_non_repeats
      endif
      
      ;;; Repair tplot variable and print summary
        ;;; Remove "bad" elements of data.x, data.y and data.v
      if not keyword_set(FORCE_NAN) then begin
        x = data.x[i_keep]
        y_size = size(data.y,/n_dimensions)
          if y_size eq 1 then begin
            y = data.y[i_keep]
            if tag_exist(data,'v') then v = data.v[i_keep]
            if tag_exist(data,'v1') then begin
              v1 = data.v1[i_keep]
              v2 = data.v2[i_keep]
            endif
          endif else begin
            y = data.y[i_keep,*]
            if tag_exist(data,'v') then v = data.v[i_keep,*]
            if tag_exist(data,'v1') then begin
              v1 = data.v1[i_keep,*]
              v2 = data.v2[i_keep,*]
            endif
          endelse
        ;;; Replace "bad" elements of data.y and data.v with NaN
      endif else begin
        x = data.x
        i_all = lindgen(n_elements(x))
        i_replace = ssl_set_complement([i_keep],[i_all])
        if min(i_replace) ne -1 then begin
          y_size = size(data.y,/n_dimensions)
            if y_size eq 1 then begin
              y = data.y
              y[i_replace] = 'NaN'
              if tag_exist(data,'v') then begin
                v = data.v
                v[i_replace] = 'NaN'
              endif
              if tag_exist(data,'v1') then begin
                v1 = data.v1
                v1[i_replace] = 'NaN'
                v2 = data.v2
                v2[i_replace] = 'NaN'
              endif
            endif else begin
              y = data.y
              y[i_replace,*] = 'NaN'
              if tag_exist(data,'v') then begin
                v = data.v
                v[i_replace,*] = 'NaN'
              endif
              if tag_exist(data,'v1') then begin
                v1 = data.v1
                v1[i_replace,*] = 'NaN'
                v2 = data.v2
                v2[i_replace,*] = 'NaN'
              endif
            endelse
          endif
       endelse
      
      ;;; Write modified data to tplot variable
      if (tag_exist(data,'v') or tag_exist(data,'v1')) then begin
        if tag_exist(data,'v') then store_data,tplot_var,data={x:x,y:y,v:v},dlimits=dlimits,limits=limits
        if tag_exist(data,'v1') then store_data,tplot_var,data={x:x,y:y,v1:v1,v2:v2},dlimits=dlimits,limits=limits
      endif else store_data,tplot_var,data={x:x,y:y},dlimits=dlimits,limits=limits
      
      ;;; Determine percentage of elements removed or replaced, and report them
      percent_removed = (n_elements(time_array) - n_elements(i_keep)) / float(n_elements(time_array)) * float(100)
      if not keyword_set(FORCE_NAN) then begin
        dprint,tplot_var,':'+strjoin(strarr(max_var_len-strlen(tplot_var))+' ')+'FAIL!, repaired: '+ strtrim(percent_removed,2) + ' % removed'
      endif else dprint,tplot_var,':'+strjoin(strarr(max_var_len-strlen(tplot_var))+' ')+'FAIL!, repaired: '+ strtrim(percent_removed,2) + ' % replaced with NaN'   
      if percent_removed gt 10d then message,'********** Warning: Over 10% removed or replaced from '+strtrim(tplot_var,2)+' **********',/continue
    endif else dprint,tplot_var,':'+strjoin(strarr(max_var_len-strlen(tplot_var))+' ')+'FAIL!'
  endelse

endfor  ;;; end of loop over tplot variables
end