;+
; NAME: thm_crib_tplot_annotation
; 
; PURPOSE:  Crib to demonstrate tplot annotation commands  
;           You can run this crib by typing:
;           IDL>.compile thm_crib_tplot_annotation
;           IDL>.go
;           
;           When you reach a stop, press
;           IDL>.c
;           to continue
;           
;           Or you can copy and paste commands directly onto the command line
;
; SEE ALSO: thm_crib_tplot.pro  (basic tplot commands)
;           thm_crib_tplot_layout.pro  (how to arrange plots within a window, and data within a plot)
;           thm_crib_tplot_range.pro   (how to control the range and scaling of plots)
;           thm_crib_tplot_export_print.pro (how to export images of plots into pngs and postscripts)
;
;
; NOTES:
;  1.  As a rule of thumb, "tplot_options" controls settings that are global to any tplot
;   "options" controls settings that are specific to a tplot variable
;   
;  2.  If you see any useful commands missing from these cribs, please let us know.
;   these cribs can help double as documentation for tplot.
;
; $LastChangedBy: aaflores $
; $LastChangedDate: 2011-10-26 17:21:28 -0700 (Wed, 26 Oct 2011) $
; $LastChangedRevision: 9206 $
; $URL: svn+ssh://thmsvn@ambrosia.ssl.berkeley.edu/repos/ssl_general/trunk/examples/thm_crib_tplot_annotation.pro $
;-



;this line deletes data so we start the crib fresh
store_data,'*',/delete

;first we set a time and load some data.
timespan,'2008-03-23'

;loading spectral data
st_swea_load, /all

;loading line plot data (stereo moments)
st_part_moments, probe='a', /get_mom

st_position_load, probe='a'

;set new color scheme (for aesthetics)
init_crib_colors

;make sure we're using window 0
tplot_options, window=0
window, 0, xsize=700, ysize=800

;increasing the xmargin so it is easier to see the labels
tplot_options, 'xmargin', [18,12] ;18 characters on left side, 12 on right
tplot_options, 'ymargin', [8,4]   ;8 characters on the bottom, 4 on the top


;basic plot for comparision
tplot,['sta_SWEA_en','sta_SWEA_mom_flux']


print,'  This first plot is the default, for reference. 
print,'Type ".c" to continue crib examples.'
stop

;use "title" to set the plot window title
tplot_options,title='HELLO WORLD'
tplot ;this just replots the previous plot with the new options


print,'  Set the plot title with "title"'
print,'Type ".c" to continue'
stop

;use x/y/ztitle & x/y/zsubtitle, to set individual axis titles
;by default, the variable name is used
options,'sta_SWEA_en',ytitle="I'm a ytitle",ztitle="I'm a ztitle"

;use options to set the ysubtitle
options,'sta_SWEA_mom_flux',ysubtitle="I'm a ysubtitle",xtitle="I'm an xtitle"

;replot
tplot


print,'  Use x/y/ztitle & x/y/zsubtitle, to set individual axis titles'
print,'Type ".c" to continue'
stop

;increase the whole plot charsize using "charsize"

tplot_options,'charsize',1.2  ;this value is a multiple of the default character size

tplot


print,'  Set the global character size using "charsize"'
print,'Type ".c" to continue'
stop

;change the charsize for each axis individually using xcharsize,ycharsize

options,'sta_SWEA_mom_flux',xcharsize=.6,ycharsize=1.4
options,'sta_SWEA_en',ycharsize=1.0
tplot


print,'  Change the charsize for each axis individually using "x/ycharsize"'
print,'Type ".c" to continue'
stop

;resetting sizes to more managable values
options,'sta_SWEA_mom_flux',xcharsize=1.0,ycharsize=1.0,xtitle=''

;use "labels" to set labels for a line plot

options,'sta_SWEA_mom_flux',labels=['Xcomp','Ycomp','Zcomp'] ;number of elements in labels should match number of components in line plot

tplot


print,'  Use "labels" to set labels for a line plot'
print,'Type ".c" to continue'
stop

;zooming in so feature is more visible
tlimit,'2008-03-23/12:00:00','2008-03-23/20:00:00'

;setting labflag will put axis labels where lines end, instead of evenly spacing labels along the axis
options,'sta_SWEA_mom_flux',labflag=2

;labflag = 1 is the default for line plots
;labflag = 0 turns labels off

tplot


print,'  Use "labflag" to control positioning of labels'
print,'Type ".c" to continue'
stop

;resetting labflag
options,'sta_SWEA_mom_flux',labflag=1

;"colors" option controls line/label color

options,'sta_SWEA_mom_flux',colors=['b','m','c'] ;number of elements should match number of components
;valid values for colors include
;'x','m','b','c','g','y','r','w', 'd','z', and 0-255
;'x' or 0 is black
;'m' or 1 is magenta
;'b' or 2 is blue
;'c' or 3 is cyan
;'g' or 4 is green
;'y' or 5 is yellow
;'r' or 6 is red
;'w' or 255 is white
;'d' is foreground color(!p.color)
;'z' is background color(!p.background)
;10-255 are elements in a continuous color table. (The default is a basic rainbow table)

tplot


print,'  "colors" option controls line/label color'
print,'Type ".c" to continue'
stop
;reset colors
options,'sta_SWEA_mom_flux',colors=[2,4,6]

;use !p.color and !p.background to change background & foreground colors
;can only use numerical color indexes (not letters)

!p.color = 1
!p.background=3

tplot


print,'  "!p.color" and "!p.background" can change foreground & background colors
print,'Type ".c" to continue'
stop

;resetting foreground & background
!p.color = 0
!p.background = 255

;label the x-axis using a single variable(distance in re)

calc,'"sta_pos_re" = "sta_pos_GSE"/6374.4',/verbose ;convert km into RE
calc,'"sta_dist_re" = sqrt(total(abs("sta_pos_re"),2))',/verbose  ;euclidean norm

options,'sta_dist_re',ytitle="Dist(RE)"  ;ytitle is used to label variables

tplot,var_label='sta_dist_re'


print,'  "var_label" can be used to label the x-axis 
print,'Type ".c" to continue'
stop

;label the x-axis using multiple single variables(state position gsm in re)

split_vec,'sta_pos_re' ;split tplot variable into individual components

;used to set label for var_label option
options,'sta_pos_re_x',ytitle='X Pos(RE)'
options,'sta_pos_re_y',ytitle='Y Pos(RE)'
options,'sta_pos_re_z',ytitle='Z Pos(RE)'

tplot,var_label=['sta_pos_re_x','sta_pos_re_y','sta_pos_re_z']


print,'  "var_label" can be used to label with multiple components
print,'Type ".c" to continue'
stop

;use x/y ticklen to control length of major ticks 

options,'sta_SWEA_mom_flux',xticklen=.15,yticklen=.05 ;value is a proportion of panel occupied by ticks, ie 1 = 100%, so we create a grid with this call
options,'sta_SWEA_en',xticklen=.20,yticklen=.25 ;these ticks are 25% and 35% of the panel size, respectively

;var_label = '' turns off variable labels
tplot,var_label=''


print, ' "psym" can be used to plot symbols at each datapoint,'
print, ' each point is plotted with a symbol so we will first decimate the data for this example' 
print,'Type ".c" to begin'
stop

;decimate data to 100 points for the whole day
tr = gettime() + [ 0, 24*3600.] 
tinterpol_mxn, 'sta_SWEA_mom_flux', interpol(tr,100), suffix='_low'

options,'sta_SWEA_mom_flux_low', psym = 4

;other options (use -# to also draw a line between points)
; 1 Plus sign (+)
; 2 Asterisk (*) 
; 3 Period (.)  
; 4 Diamond  
; 5 Triangle
; 6 Square  
; 7 X  
; 8 User-defined (See examples below)  
; 10 Histogram mode

tplot,['sta_SWEA_en','sta_SWEA_mom_flux_low']


print, ' Use "thm_ui_set_symbol" to set the user defined symbol'
print, ' the "size" and "fill" keywords can be used to control the symbol size and whether they are filled' 
print,'Type ".c" to continue'
stop

;set the symbol to fill triagles
thm_ui_set_symbol, 5, /fill

; other custom options
; 1 Plus sign
; 2 Star
; 3 Circle
; 4 Diamond
; 5 Triangle
; 6 Square
; 7 X

;set the variable to use the custom symbol
options, 'sta_SWEA_mom_flux_low', psym=8

tplot


print, ' call "thm_ui_set_symbol" again to change the symbol or its properties'
print,'Type ".c" to continue'
stop

;set custom symbol to larger X's
thm_ui_set_symbol, 7, size=2

tplot


print, ' return to the original variables'
print,'Type ".c" to continue'
stop

tplot,['sta_SWEA_en','sta_SWEA_mom_flux']


print,'  "x/yticklen" can be used to control tick length'
print,'Type ".c" to continue'
stop

;use ticklen to create a grid
options,'sta_SWEA_mom_flux',xticklen=1,yticklen=1 ;value is a proportion of panel occupied by ticks, ie 1 = 100%, so we create a grid with this call
options,'sta_SWEA_en',xticklen=1,yticklen=1;these ticks are 25% and 35% of the panel size, respectively

tplot


print,'You can set long ticks to create a grid on major ticks.'
print,'Type ".c" to continue'
stop

;resetting tick-len
options,'sta_SWEA_mom_flux',xticklen=.1,yticklen=.05
options,'sta_SWEA_en',xticklen=.1,yticklen=.05

;Control thickness of borders & ticks with y/x-thick

options,'sta_SWEA_mom_flux',xthick=2.0,ythick=2.0
options,'sta_SWEA_en',xthick=5.0,ythick=1.0

tplot


print,'You can control border/tick/grid thickness using x/y thick'
print,'Type ".c" to continue'
stop

;Control line thickness using 'thick'
options,'sta_SWEA_mom_flux',thick=2.0

tplot


print,'You can control line thickness using thick'
print,'Type ".c" to continue'
stop

;You can create your own custom annotations using the get_plot_pos keyword to return the corners of the plots generated by tplot in normalized coordinates.
tplot,['sta_SWEA_mom_flux','sta_SWEA_en'],get_plot_pos=p


;p is a 4xNpanels variable.  Each 4-element segment is p[*,panel_num] = [lower_left_corner_x,lower_left_corner_y,upper_right_corner_x,upper_right_corner_y]
;You can use this with built in IDL routines like plots to add your own lines.  
plots,[p[0,0],p[2,0]],[p[1,0],p[3,0]],/normal  ;Draws a line from the lower left corner of the top panel to the upper right of the top panel
print,'You can create custom annotations with the help from get_plot_pos'

print,"We're done!"

end
