;+
;Procedure: ACE_MAG_SWEPAM_LOAD
;
;Purpose:  Loads ACE Magnetometer Values in GSM coordinates (1 minute averages) and/or
;          Solar Wind Electron Proton Alpha Monitor data
;
;keywords:
;   datatype = (Optional) type of data ('mag', 'swepam', 'all'), default is 'all' and includes
;              all the ascii type data files for ACE (mag and swepam)
;   trange = (Optional) Time range of interest (2 element array), default is tplot trange
;   downloadonly = (Optional) Set this to 1 to retrieve the data but not load it into
;                             tplot variables
;   verbose = (Optional) Set this keyword to get more messages, default is off
;   source = (Optional) for directories different from the default ACE data directory
;   tplotnames = (Optional) tplot variable names
;
;Example:
;   ace_mag_swepam_load
;Notes:
;
; Author: Cindy Goethel
;
; $LastChangedBy: cgoethel $
; $LastChangedDate: 2008-04-21 11:04:17 -0700 (Mon, 21 Apr 2008) $
; $LastChangedRevision: 2794 $
; $URL $
;-

pro ace_mag_swepam_load, datatype=datatype, trange=trange, downloadonly=downloadonly, $
                         addmaster=addmaster, verbose=verbose, tplotnames=tn, source_options=source

  ace_init

  if not keyword_set(source) then source = !ace
  if not keyword_set(datatype) then datatype = 'all'
  if keyword_set(datatype) then Begin
      if datatype ne 'mag' and datatype ne 'swepam' and datatype ne 'all' then Begin
        print, 'Incorrect datatype specified.'
        print, "Valid datatypes for ACE ascii data include: ('mag', 'swepam', 'all')"
        if obj_valid(progobj) then progobj -> update, 0.0, $
             text = string("Incorrect datatype specified - valid datatypes are ('mag', 'swepam', 'all')")
        return
      endif
  endif
  
  datatype = strlowcase(datatype)
  if not keyword_set(trange) then trange = timerange(/current)

  if datatype ne 'all' then Begin
      pathformat = 'ace/'+datatype+'/k0/YYYY/YYYYMMDD_ace_'+datatype+'_1m.txt'
      relpathnames = file_dailynames(file_format=pathformat,trange=trange,addmaster=addmaster)
  endif else Begin
      pathformat = 'ace/mag/k0/YYYY/YYYYMMDD_ace_mag_1m.txt'
      magpathnames = file_dailynames(file_format=pathformat,trange=trange,addmaster=addmaster)
      pathformat = 'ace/swepam/k0/YYYY/YYYYMMDD_ace_swepam_1m.txt'
      swepathnames = file_dailynames(file_format=pathformat,trange=trange,addmaster=addmaster)
      relpathnames = [magpathnames, swepathnames]
  endelse

  files = file_retrieve(relpathnames, local_data_dir=!ace.local_data_dir, $
                            remote_data_dir=!ace.remote_data_dir)

  if keyword_set(downloadonly) then return

  n_files=n_elements(files)
  for i=0, n_files-1 do Begin
     if file_test(files(i)) eq 0 then Begin
         dprint,dlevel=1, verbose=verbose, string('ACE file not found: '+files(i))
         if obj_valid(progobj) then progobj -> update, 0.0, $
             text = string('ACE file not found: '+files(i))
         print, 'ACE file not found: '+files(i)
     endif else Begin
         print, 'ACE file '+files(i)+' retrieved.'
         if obj_valid(progobj) then progobj -> update, 0.0, $
              text = string('ACE file '+files(i)+' retrieved.')
         if (strpos(files(i), 'mag') ne -1) then Begin
            filetype='mag'
            data_start=20
         endif else Begin
            filetype='swepam'
            data_start=18
         endelse
         data=read_ascii(files(i), data_start=data_start)
         print, 'Loading ACE file '+files(i)
         if obj_valid(progobj) then progobj -> update, 0.0, $
              text = string('Loading ACE file '+files(i))         
         ace2tplot, data, filetype
     endelse
  endfor

  print, 'Done loading ACE files.'
  if obj_valid(progobj) then progobj -> update, 0.0, $
        text = string('Done loading ACE files.')         

end

pro ace2tplot, data, data_type

  ; create an array of times (time format is double)
  time_array = fix(data.field01[0:5,*])
  ace_tstring = create_ace_tstring(time_array)
  ace_time = reform(time_double(ace_tstring))
  
  ; create variable names and store data
  if (data_type eq 'mag') then Begin
    fields = transpose(reform(data.field01[7:9,*]))
    store_data,'ace_mag',dat={x:(ace_time), y:fields},$
                dlim={ytitle:'Fields, nT', coord:'GSM'}
    store_data,'ace_mag_bt',dat={x:ace_time, y:reform(data.field01[10,*])}
    store_data,'ace_mag_lat',dat={x:ace_time, y:reform(data.field01[11,*])},$
                dlim={ytitle:'Lat, deg'}
    store_data,'ace_mag_lon',dat={x:ace_time, y:reform(data.field01[12,*])},$
                dlim={ytitle:'Lon, deg'}
    store_data,'ace_mag_stat',dat={x:ace_time, y:reform(data.field01[6,*])}
  endif else Begin
    store_data,'ace_swepam_pden',dat={x:ace_time, y:reform(data.field01[7,*])}
    store_data,'ace_swepam_bspd',dat={x:ace_time, y:reform(data.field01[8,*])}
    store_data,'ace_swepam_itmp',dat={x:ace_time, y:reform(data.field01[9,*])}
    store_data,'ace_swepam_stat',dat={x:ace_time, y:reform(data.field01[6,*])}
  endelse

end

function create_ace_tstring,time_array

    ntimes = n_elements(time_array(0,*))
    timestr = make_array(6, ntimes, /string)
    timestr=strcompress(string(time_array))

    ; deal with hours and minutes (parse the hhmm string)
    for i=0, ntimes-1 do Begin
         if (time_array(3,i) lt 100)then Begin
           timestr(3,i)= '0'
           if (time_array(3,i) lt 10) then ndigits=1 else ndigits=2
           hm = strcompress(string(time_array(3,i)))
           timestr(4,i)=strmid(hm, 1, ndigits)
         endif
         if (time_array(3,i) ge 100 and time_array(3,i) lt 1000) then Begin
           hm = strcompress(string(time_array(3,i)))
           timestr(3,i)=strmid(hm, 1, 1)
           timestr(4,i)=strmid(hm, 2, 2)
        endif
        if (time_array(3,i) ge 1000) then Begin
           hm = strcompress(string(time_array(3,i)))
           timestr(3,i)=strmid(hm, 1, 2)
           timestr(4,i)=strmid(hm, 3, 2)
        endif
    endfor

    ; parse year, month, day
    year = strmid(timestr(0,*), 1, 4)
    month = strmid(timestr(1,*), 1, strlen(timestr(1,*)))
    day = strmid(timestr(2,*), 1, strlen(timestr(2,*)))

    ;create the time string
    tstring = strcompress(year+'-'+month+'-'+day+'/'+ timestr(3,*)+':'+timestr(4,*))

    return, tstring

end

pro ace_init, reset=reset, local_data_dir=local_data_dir, remote_data_dir=remote_data_dir

    defsysv,'!ace',exists=exists
    if not keyword_set(exists) then begin
       defsysv,'!ace',  file_retrieve(/structure_format)
    endif

    if keyword_set(reset) then !ace.init=0

    if !ace.init ne 0 then return

    !ace = file_retrieve(/structure_format)
    !ace.local_data_dir = root_data_dir()
    !ace.remote_data_dir = 'http://themis.ssl.berkeley.edu/data/'

    if file_test(!ace.local_data_dir+'ace/.master') then !ace.no_download=1  ; Local directory IS the master directory

    if keyword_set(name) then call_procedure,name

    !ace.init = 1

end
