;+
; NAME: rbsp_efw_cal_xspec
;
; PURPOSE: Calibrate RBSP cross-spectral data
;
; NOTES: meant to be called from rbsp_load_efw_xspec.pro
;
;	The default is that xspec0 is SCMw x E12AC
;	and xspec1 is V1AC x V2AC
;	xspec3 and xspec4 are not used at this time
;
;	src1 and src2 refer to the individual sources that comprise the xspec
;
;
;	Data calibration
;		subtract offset
;			src1_2 = (double(src1.y) - offset_src1)
;		Compensate for dynamic range compression
;			src1_2 /= 8.
;		Divide the 0th bin by 4 (from weirdness doc)
;			src1_2[*,0] /= 4.
;		Compensate for using Hanning window
;			src1_2 /= 0.375^2
;		Apply ADC conversion
;			src1_2 *= adc_factor^2
;		Apply channel gain
;			src1_2 *= gain_src1^2
;		Resulting data in V^2/Hz
;
;
;
; SEE ALSO:
;
; HISTORY:
;   2012-xx-xx: Created by Aaron W Breneman, University of Minnesota
;
; VERSION:
; $LastChangedBy:  $
; $LastChangedDate:  $
; $LastChangedRevision:  $
; $URL: svn+ssh://thmsvn@ambrosia.ssl.berkeley.edu/repos/ssl_general/trunk/missions/rbsp/efw/rbsp_efw_cal_fbk.pro $
;
;-


pro rbsp_efw_cal_xspec, probe=probe, trange=trange, datatype=datatype


if datatype eq 'xspec' then begin

	
	compile_opt idl2
	
	;dprint,verbose=verbose,dlevel=4,'$Id: rbsp_efw_cal_filterbank.pro 10920 2012-09-17 17:08:00Z aaron_breneman $'
	
	if ~keyword_set(trange) then trange = timerange()
	
	cp0 = rbsp_efw_get_cal_params(trange[0])
	
	

	; first see if we have 36, 64, or 112 bins
	names=''
	nn=''
;	names=tnames('rbsp'+probe+'_efw_xspec_'+'??' + '_xspec?_src')



;For now assume 64 bins
	names=tnames('rbsp'+probe+'_efw_xspec_64_xspec?_src*')
	bins = 64
;	print,names	
;	print,'rbsp'+probe+'_efw_xspec_'+'??' + '_xspec?_src'


;	if strlen(names[0]) ne 0 then nn=strmid(names[0],10,1)
;	if nn eq '3' then bins='36' $
;		else if nn eq '6' then bins='64' $
;		else if nn eq '1' then bins='112' $
;		else bins='0'
	
	
		
	
	
	adc_factor = 2.5d / 32767.5d
	
	
	case strlowcase(probe) of
	  'a': cp = cp0.a
	  'b': cp = cp0.b
	  else: dprint, 'Invalid probe name. Calibration aborted.'
	endcase
	
;	boom_length = cp.boom_length
;	boom_shorting_factor = cp.boom_shorting_factor
	
	rbspx = 'rbsp' + probe[0]
	

 

	;162B   -> sqrt symbol for window plotting
	;326B   -> ''   ''       ''  ps plotting
	
	;Print version of units
;	ssym_ps = '!9' + string("326B) +'!3'+'Hz' ;sqrt symbol
	ssym_ps = 'Hz'

	;ssym_win = '!9' + string("162B) +'!3'+'Hz' ;sqrt symbol
	units = replicate('V^2/Hz',8)	
	
	

;--------------------------------------------------------------------	
;Calibrate xspec0  (src1 = SCMw, src2 = E12AC)
	
	get_data,rbspx + '_efw_xspec_64_xspec0_src1',data=src1,dlimits=dlim_src1
	get_data,rbspx + '_efw_xspec_64_xspec0_src2',data=src2,dlimits=dlim_src2
	get_data,rbspx + '_efw_xspec_64_xspec0_rc',data=rc,dlimits=dlim_rc
	get_data,rbspx + '_efw_xspec_64_xspec0_ic',data=ic,dlimits=dlim_ic
	

	offset_src1 = cp.ADC_offset_MSC[2]
	offset_src2 = cp.ADC_offset_EAC[0]
	offset_rc = 0.
	offset_ic = 0.  ;what should these be set to?

	gain_src1 = cp.ADC_gain_MSC[2]/adc_factor
	gain_src2 = cp.ADC_gain_EAC[0]/adc_factor
	gain_rc = sqrt(cp.ADC_gain_MSC[2]*cp.ADC_gain_EAC[0])/adc_factor
	gain_ic = gain_rc



	;subtract offset
	src1_2 = (double(src1.y) - offset_src1)
	;Compensate for 4 bit dynamic range compression
	src1_2 /= 8.
	;Divide the 0th bin by 4 (from weirdness doc)
	src1_2[*,0] /= 4.
	;Compensate for using Hanning window
	src1_2 /= 0.375^2
	;Apply ADC conversion
	src1_2 *= adc_factor^2
	;Apply channel gain
	src1_2 *= gain_src1^2



	;subtract offset
	src2_2 = (double(src2.y) - offset_src2)
	;Compensate for 4 bit dynamic range compression
	src2_2 /= 8.
	;Divide the 0th bin by 4 (from weirdness doc)
	src2_2[*,0] /= 4.
	;Compensate for using Hanning window
	src2_2 /= 0.375^2
	;Apply ADC conversion
	src2_2 *= adc_factor^2
	;Apply channel gain
	src2_2 *= gain_src2^2


	;subtract offset
	rc_2 = (double(rc.y) - offset_rc)
	;Compensate for 4 bit dynamic range compression
	rc_2 /= 8.
	;Divide the 0th bin by 4 (from weirdness doc)
	rc_2[*,0] /= 4.
	;Compensate for using Hanning window
	rc_2 /= 0.375^2
	;Apply ADC conversion
	rc_2 *= adc_factor^2
	;Apply channel gain
	rc_2 *= gain_rc^2



	;subtract offset
	ic_2 = (double(ic.y) - offset_ic)
	;Compensate for 4 bit dynamic range compression
	ic_2 /= 8.
	;Divide the 0th bin by 4 (from weirdness doc)
	ic_2[*,0] /= 4.
	;Compensate for using Hanning window
	ic_2 /= 0.375^2
	;Apply ADC conversion
	ic_2 *= adc_factor^2
	;Apply channel gain
	ic_2 *= gain_ic^2



	dlim_src1.data_att.units = 'V^2/'+ssym_ps
	dlim_src2.data_att.units = 'V^2/'+ssym_ps
	dlim_rc.data_att.units = 'V^2/'+ssym_ps
	dlim_ic.data_att.units = 'V^2/'+ssym_ps

	store_data,rbspx + '_efw_xspec_64_xspec0_src1'+'_cal',data={x:src1.x,y:src1_2,v:src1.v},dlim=dlim_src1
	store_data,rbspx + '_efw_xspec_64_xspec0_src2'+'_cal',data={x:src2.x,y:src2_2,v:src2.v},dlim=dlim_src2
	store_data,rbspx + '_efw_xspec_64_xspec0_rc'+'_cal',data={x:rc.x,y:rc_2,v:rc.v},dlim=dlim_rc
	store_data,rbspx + '_efw_xspec_64_xspec0_ic'+'_cal',data={x:ic.x,y:ic_2,v:ic.v},dlim=dlim_ic
	
	
	options,rbspx + '_efw_xspec_64_xspec0_src1'+'_cal',labels = 'Xspec SCMW'
	options,rbspx + '_efw_xspec_64_xspec0_src1'+'_cal','ysubtitle','[Hz]'
	options,rbspx + '_efw_xspec_64_xspec0_src1'+'_cal',labflag = 1
	options,rbspx + '_efw_xspec_64_xspec0_src1'+'_cal','ztitle','V/'+ssym_ps
	options,rbspx + '_efw_xspec_64_xspec0_src1'+'_cal','ytitle','SCMw'


	options,rbspx + '_efw_xspec_64_xspec0_src2'+'_cal',labels = 'Xspec E12AC'
	options,rbspx + '_efw_xspec_64_xspec0_src2'+'_cal','ysubtitle','[Hz]'
	options,rbspx + '_efw_xspec_64_xspec0_src2'+'_cal',labflag = 1
	options,rbspx + '_efw_xspec_64_xspec0_src2'+'_cal','ztitle','V/'+ssym_ps
	options,rbspx + '_efw_xspec_64_xspec0_src2'+'_cal','ytitle','E12AC'

		
	options,rbspx + '_efw_xspec_64_xspec0_rc'+'_cal',labels = 'Xspec SCMW x E12AC'
	options,rbspx + '_efw_xspec_64_xspec0_rc'+'_cal','ysubtitle','[Hz]'
	options,rbspx + '_efw_xspec_64_xspec0_rc'+'_cal',labflag = 1
	options,rbspx + '_efw_xspec_64_xspec0_rc'+'_cal','ztitle','V/'+ssym_ps
	options,rbspx + '_efw_xspec_64_xspec0_rc'+'_cal','ytitle','RE(SCMW x E12AC)'

	options,rbspx + '_efw_xspec_64_xspec0_ic'+'_cal',labels = 'Xspec SCMW x E12AC'
	options,rbspx + '_efw_xspec_64_xspec0_ic'+'_cal','ysubtitle','[Hz]'
	options,rbspx + '_efw_xspec_64_xspec0_ic'+'_cal',labflag = 1
	options,rbspx + '_efw_xspec_64_xspec0_ic'+'_cal','ztitle','V/'+ssym_ps
	options,rbspx + '_efw_xspec_64_xspec0_ic'+'_cal','ytitle','IM(SCMW x E12AC)'


	
	
;--------------------------------------------------------------------	
;Calibrate xspec1  (src1 = V1AC, src2 = V2AC)
	
	get_data,rbspx + '_efw_xspec_64_xspec1_src1',data=src1,dlimits=dlim_src1
	get_data,rbspx + '_efw_xspec_64_xspec1_src2',data=src2,dlimits=dlim_src2
	get_data,rbspx + '_efw_xspec_64_xspec1_rc',data=rc,dlimits=dlim_rc
	get_data,rbspx + '_efw_xspec_64_xspec1_ic',data=ic,dlimits=dlim_ic
	

	offset_src1 = cp.ADC_offset_VAC[0]
	offset_src2 = cp.ADC_offset_VAC[1]
	offset_rc = 0.
	offset_ic = 0.  ;what should these be set to?

	gain_src1 = cp.ADC_gain_VAC[0]/adc_factor
	gain_src2 = cp.ADC_gain_VAC[1]/adc_factor
	gain_rc = sqrt(cp.ADC_gain_VAC[0]*cp.ADC_gain_VAC[1])/adc_factor
	gain_ic = gain_rc



	;subtract offset
	src1_2 = (double(src1.y) - offset_src1)
	;Compensate for 4 bit dynamic range compression
	src1_2 /= 8.
	;Divide the 0th bin by 4 (from weirdness doc)
	src1_2[*,0] /= 4.
	;Compensate for using Hanning window
	src1_2 /= 0.375^2
	;Apply ADC conversion
	src1_2 *= adc_factor^2
	;Apply channel gain
	src1_2 *= gain_src1^2



	;subtract offset
	src2_2 = (double(src2.y) - offset_src2)
	;Compensate for 4 bit dynamic range compression
	src2_2 /= 8.
	;Divide the 0th bin by 4 (from weirdness doc)
	src2_2[*,0] /= 4.
	;Compensate for using Hanning window
	src2_2 /= 0.375^2
	;Apply ADC conversion
	src2_2 *= adc_factor^2
	;Apply channel gain
	src2_2 *= gain_src2^2



	;subtract offset
	rc_2 = (double(rc.y) - offset_rc)
	;Compensate for 4 bit dynamic range compression
	rc_2 /= 8.
	;Divide the 0th bin by 4 (from weirdness doc)
	rc_2[*,0] /= 4.
	;Compensate for using Hanning window
	rc_2 /= 0.375^2
	;Apply ADC conversion
	rc_2 *= adc_factor^2
	;Apply channel gain
	rc_2 *= gain_rc^2



	;subtract offset
	ic_2 = (double(ic.y) - offset_ic)
	;Compensate for 4 bit dynamic range compression
	ic_2 /= 8.
	;Divide the 0th bin by 4 (from weirdness doc)
	ic_2[*,0] /= 4.
	;Compensate for using Hanning window
	ic_2 /= 0.375^2
	;Apply ADC conversion
	ic_2 *= adc_factor^2
	;Apply channel gain
	ic_2 *= gain_ic^2




	dlim_src1.data_att.units = 'V^2/'+ssym_ps
	dlim_src2.data_att.units = 'V^2/'+ssym_ps
	dlim_rc.data_att.units = 'V^2/'+ssym_ps
	dlim_ic.data_att.units = 'V^2/'+ssym_ps

	store_data,rbspx + '_efw_xspec_64_xspec1_src1'+'_cal',data={x:src1.x,y:src1_2,v:src1.v},dlim=dlim_src1
	store_data,rbspx + '_efw_xspec_64_xspec1_src2'+'_cal',data={x:src2.x,y:src2_2,v:src2.v},dlim=dlim_src2
	store_data,rbspx + '_efw_xspec_64_xspec1_rc'+'_cal',data={x:rc.x,y:rc_2,v:rc.v},dlim=dlim_rc
	store_data,rbspx + '_efw_xspec_64_xspec1_ic'+'_cal',data={x:ic.x,y:ic_2,v:ic.v},dlim=dlim_ic
	
	

	options,rbspx + '_efw_xspec_64_xspec1_src1'+'_cal',labels = 'Xspec SCMW'
	options,rbspx + '_efw_xspec_64_xspec1_src1'+'_cal','ysubtitle','[Hz]'
	options,rbspx + '_efw_xspec_64_xspec1_src1'+'_cal',labflag = 1
	options,rbspx + '_efw_xspec_64_xspec1_src1'+'_cal','ztitle','V/'+ssym_ps
	options,rbspx + '_efw_xspec_64_xspec1_src1'+'_cal','ytitle','SCMw'


	options,rbspx + '_efw_xspec_64_xspec1_src2'+'_cal',labels = 'Xspec E12AC'
	options,rbspx + '_efw_xspec_64_xspec1_src2'+'_cal','ysubtitle','[Hz]'
	options,rbspx + '_efw_xspec_64_xspec1_src2'+'_cal',labflag = 1
	options,rbspx + '_efw_xspec_64_xspec1_src2'+'_cal','ztitle','V/'+ssym_ps
	options,rbspx + '_efw_xspec_64_xspec1_src2'+'_cal','ytitle','E12AC'

		
	options,rbspx + '_efw_xspec_64_xspec1_rc'+'_cal',labels = 'Xspec SCMW x E12AC'
	options,rbspx + '_efw_xspec_64_xspec1_rc'+'_cal','ysubtitle','[Hz]'
	options,rbspx + '_efw_xspec_64_xspec1_rc'+'_cal',labflag = 1
	options,rbspx + '_efw_xspec_64_xspec1_rc'+'_cal','ztitle','V/'+ssym_ps
	options,rbspx + '_efw_xspec_64_xspec1_rc'+'_cal','ytitle','RE(SCMW x E12AC)'

	options,rbspx + '_efw_xspec_64_xspec1_ic'+'_cal',labels = 'Xspec SCMW x E12AC'
	options,rbspx + '_efw_xspec_64_xspec1_ic'+'_cal','ysubtitle','[Hz]'
	options,rbspx + '_efw_xspec_64_xspec1_ic'+'_cal',labflag = 1
	options,rbspx + '_efw_xspec_64_xspec1_ic'+'_cal','ztitle','V/'+ssym_ps
	options,rbspx + '_efw_xspec_64_xspec1_ic'+'_cal','ytitle','IM(SCMW x E12AC)'

	ylim,rbspx + '_efw_xspec_64_xspec1_ic'+'_cal',3,10000,1
	ylim,rbspx + '_efw_xspec_64_xspec1_ic'+'_cal',3,10000,1
	ylim,rbspx + '_efw_xspec_64_xspec1_ic'+'_cal',3,10000,1
	ylim,rbspx + '_efw_xspec_64_xspec1_ic'+'_cal',3,10000,1
	ylim,rbspx + '_efw_xspec_64_xspec1_ic'+'_cal',3,10000,1

	zlim,rbspx + '_efw_xspec_64_xspec1_ic'+'_cal',0.001^2,0.1^2,1
	zlim,rbspx + '_efw_xspec_64_xspec1_ic'+'_cal',0.001^2,0.1^2,1
	zlim,rbspx + '_efw_xspec_64_xspec1_ic'+'_cal',0.001^2,0.1^2,1
	zlim,rbspx + '_efw_xspec_64_xspec1_ic'+'_cal',0.001^2,0.1^2,1
	zlim,rbspx + '_efw_xspec_64_xspec1_ic'+'_cal',0.001^2,0.1^2,1

	
	
	
	endif else begin
	      print, ''
      dprint, 'Invalid datatype. Calibration aborted.'
      print, ''
     return


endelse



;Fix the bin labels


if bins eq 36 then begin

	tplot_names,'rbsp'+probe+'_efw_xspec_36_xspec?_*',names=tnames
	
	; 36 bin spec
	fbins_36 = [findgen(8)*8,findgen(4)*16+64,$
			findgen(4)*32+128,findgen(4)*64+256,$
			findgen(4)*128+512,findgen(4)*256+1024,$
			findgen(4)*512+2048,findgen(4)*1024+4096]
	
	fcenter_36=(fbins_36[0:35] + fbins_36[1:35])/2.
	
	fbin_labels_36=strarr(36)
	fbin_labels_36[0:35]=string(fbins_36[0:35], format='(I0)')+'-'+$
				string(fbins_36[1:36], format='(I0)')+' Hz'
	
	
	for i=0,size(tnames,/n_elements)-1 do begin
		get_data,tnames[i],data=d,limits=l,dlimits=dl
		d.v=fbins_36[0:35]
		store_data,tnames[i],data=d,limits=l,dlimits=dl
	endfor


endif

if bins eq 64 then begin

	tplot_names,'rbsp'+probe+'_efw_xspec_64_xspec?_*',names=tnames
	
	
	; 64 bin spec
	fbins_64=[findgen(16)*8., findgen(8)*16.+128, $
				findgen(8)*32.+256, findgen(8)*64.+512,$
				findgen(8)*128.+1024., findgen(8)*256.+2048, $
				findgen(9)*512.+4096]
	
	fcenter_64=(fbins_64[0:63] + fbins_64[1:64])/2.
	
	fbin_labels_64=strarr(64)
	fbin_labels_64[0:63]=string(fbins_64[0:63], format='(I0)')+'-'+$
				string(fbins_64[1:64], format='(I0)')+' Hz'
	
	
	for i=0,size(tnames,/n_elements)-1 do begin
		get_data,tnames[i],data=d,limits=l,dlimits=dl
		d.v=fbins_64[0:63]
		store_data,tnames[i],data=d,limits=l,dlimits=dl
	endfor
endif


if bins eq 112 then begin

	tplot_names,'rbsp'+probe+'_efw_xspec_112_xspec?_*',names=tnames
	
	
	; 112 bin spec
	fbins_112 = [findgen(32)*8,findgen(16)*16+256,$
			findgen(16)*32+512,findgen(16)*64+1024,$
			findgen(16)*128+2048,findgen(16)*256+4096]
	
	fcenter_112=(fbins_112[0:111] + fbins_112[1:112])/2.
	
	fbin_labels_112=strarr(112)
	fbin_labels_112[0:111]=string(fbins_112[0:111], format='(I0)')+'-'+$
				string(fbins_112[1:112], format='(I0)')+' Hz'
	
	
	for i=0,size(tnames,/n_elements)-1 do begin
		get_data,tnames[i],data=d,limits=l,dlimits=dl
		d.v=fbins_112[0:111]
		store_data,tnames[i],data=d,limits=l,dlimits=dl
	endfor
endif



end



