;+
; NAME: rbsp_efw_cal_spec
;
; PURPOSE: Calibrate RBSP spectral data
;
; NOTES: Intended to be called from rbsp_load_efw_spec.pro
;
;
; Calibration procedure
;
;	subtract offset
;		new_y = (double(data.y) - offset[j])
;   Compensate for 4 bit dynamic range compression
;       new_y /= 16.
;	Divide the 0th bin by 4 (from weirdness doc)
;		new_y[*,0] /= 4.
;	Compensate for using Hanning window
;		new_y /= 0.375^2
;	Apply ADC conversion
;		new_y *= adc_factor^2
;	Apply channel gain
;		new_y *= gain[j]^2
;	Apply boom length to change from V to mV/m				
;		new_y *=  conversion[j]   ;mV/m ^2 /Hz
;
;
;
; HISTORY:
;   2012-xx-xx: Created by Aaron W Breneman, University of Minnesota
;
; VERSION:
; $LastChangedBy:  $
; $LastChangedDate:  $
; $LastChangedRevision:  $
; $URL: svn+ssh://thmsvn@ambrosia.ssl.berkeley.edu/repos/ssl_general/trunk/missions/rbsp/efw/rbsp_efw_cal_fbk.pro $
;
;-


pro rbsp_efw_cal_spec, probe=probe, trange=trange, datatype=datatype


if datatype eq 'spec' then begin

	
	compile_opt idl2
	
	;dprint,verbose=verbose,dlevel=4,'$Id: rbsp_efw_cal_filterbank.pro 10920 2012-09-17 17:08:00Z aaron_breneman $'
	
	if ~keyword_set(trange) then trange = timerange()
	
	cp0 = rbsp_efw_get_cal_params(trange[0])
	
	
	
	
	; first see if we have 36, 64, or 112 bins
	names=''
	nn=''
	names=tnames('rbsp'+probe+'*spec?')
	if strlen(names[0]) ne 0 then nn=strmid(names[0],10,1)
	if nn eq '3' then bins='36' $
		else if nn eq '6' then bins='64' $
		else if nn eq '1' then bins='112' $
		else bins='0'
	
	
		
	
	
	adc_factor = 2.5d / 32767.5d
	

	case strlowcase(probe) of
	  'a': cp = cp0.a
	  'b': cp = cp0.b
	  else: dprint, 'Invalid probe name. Calibration aborted.'
	endcase
	
	boom_length = cp.boom_length
	boom_shorting_factor = cp.boom_shorting_factor
	
	rbspx = 'rbsp' + probe[0]
	
	
	tvar = rbspx + ['_efw_64_spec0',$
					'_efw_64_spec1',$
					'_efw_64_spec2',$
					'_efw_64_spec3',$
					'_efw_64_spec4',$
					'_efw_64_spec5',$
					'_efw_64_spec6']
	
	
	gain = [cp.ADC_gain_EAC[0],$
			cp.ADC_gain_EAC[2],$
			cp.ADC_gain_MSC[0],$
			cp.ADC_gain_MSC[1],$
			cp.ADC_gain_MSC[2],$
			cp.ADC_gain_VAC[0],$
			cp.ADC_gain_VAC[0]]
	
	gain = gain/adc_factor  
	
	;Convert to mV/m and nT
	conversion = [1000./boom_length[0],$
		   1000./boom_length[2],$
		   1.,$
		   1.,$
		   1.,$
		   1.,$
		   1.]
		   
			
	offset = [cp.ADC_offset_EAC[0],$
			cp.ADC_offset_EAC[2],$
			cp.ADC_offset_MSC[0],$
			cp.ADC_offset_MSC[1],$
			cp.ADC_offset_MSC[2],$
			cp.ADC_offset_VAC[0],$
			cp.ADC_offset_VAC[0]]
	
	
	label = ['E12AC','E56AC','MSCpar','MSCper','MSCw','V1AC','V2AC']
	
	
	;162B   -> sqrt symbol for window plotting
	;326B   -> ''   ''       ''  ps plotting
	
	;Print version of units
;	ssym_ps = '!9' + string("326B) +'!3'+'Hz' ;sqrt symbol
	ssym_ps = 'Hz'
	
	;ssym_win = '!9' + string("162B) +'!3'+'Hz' ;sqrt symbol
;	units = ['mV/m','mV/m','nT','nT','nT','volts','volts']	 		 
	units = ['mV/m ^2','mV/m ^2','nT^2','nT^2','nT^2','volts^2','volts^2']	 		 
	 
	
	
	ztitle = label + ' ' + units + '/' + ssym_ps
	
	for j=0,6 do begin
	
		get_data,tvar[j],data=data,dlimits=dlim
	
		if dlim.data_att.units eq 'ADC' then begin
	
		tst = size(data)
			
			;make sure there's data
			if tst[0] ne 0 then begin
		
				;new_y =  (double(data.y) - offset[j]) * 6.46751787d-10 * gain[j]^2    ;V^2/Hz

				;subtract offset
				new_y = (double(data.y) - offset[j])

				;Compensate for 4 bit dynamic range compression
				new_y /= 4.^2
				
				;Divide the 0th bin by 4 (from weirdness doc)
				new_y[*,0] /= 4.

				;Compensate for using Hanning window
				new_y /= 0.375^2
				
				;Apply ADC conversion
				new_y *= adc_factor^2
				
				;Apply channel gain
				new_y *= gain[j]^2

				;Apply boom length to change from V to mV/m				
				new_y *=  conversion[j]   ;mV/m ^2 / Hz
							
	
	
			
				new_data = {x:data.x,y:new_y,v:data.v}
				dlim.data_att.units = units[j]
				colors = [1]
				
				str_element, dlim, 'cdf', /delete
				str_element, dlim, 'code_id', /delete
				
				newname = tvar[j]
				store_data, newname,data=new_data,dlim=dlim
				
				options,newname,labels='Spec ' + label[j]
				options,newname,labflag=1
				options,newname,'ztitle',ztitle[j]
				options,newname,'ysubtitle','[Hz]'
				ylim,newname,3,10000,1
				zlim,newname,0.001^2,0.1^2,1

		
			endif	
		endif else print,tvar[j], ' is already calibrated'
	endfor



endif else begin
	print, ''
    dprint, 'Invalid datatype. Calibration aborted.'
    print, ''
    return
endelse



;Fix the bin labels

if bins eq 36 then begin

	tplot_names,'*efw_36_spec*',names=tnames
	
	
	; 36 bin spec
	fbins_36 = [findgen(8)*8,findgen(4)*16+64,$
			findgen(4)*32+128,findgen(4)*64+256,$
			findgen(4)*128+512,findgen(4)*256+1024,$
			findgen(4)*512+2048,findgen(4)*1024+4096]
	
	fcenter_36=(fbins_36[0:35] + fbins_36[1:35])/2.
	
	fbin_labels_36=strarr(36)
	fbin_labels_36[0:35]=string(fbins_36[0:35], format='(I0)')+'-'+$
				string(fbins_36[1:36], format='(I0)')+' Hz'
	
	
	for i=0,size(tnames,/n_elements)-1 do begin
		get_data,tnames[i],data=d,limits=l,dlimits=dl
		d.v=fbins_36[0:35]
		store_data,tnames[i],data=d,limits=l,dlimits=dl
	endfor


endif

if bins eq 64 then begin

	tplot_names,'*efw_64_spec*',names=tnames
	
	
	; 64 bin spec
	fbins_64=[findgen(16)*8., findgen(8)*16.+128, $
				findgen(8)*32.+256, findgen(8)*64.+512,$
				findgen(8)*128.+1024., findgen(8)*256.+2048, $
				findgen(9)*512.+4096]
	
	fcenter_64=(fbins_64[0:63] + fbins_64[1:64])/2.
	
	fbin_labels_64=strarr(64)
	fbin_labels_64[0:63]=string(fbins_64[0:63], format='(I0)')+'-'+$
				string(fbins_64[1:64], format='(I0)')+' Hz'
	
	
	for i=0,size(tnames,/n_elements)-1 do begin
		get_data,tnames[i],data=d,limits=l,dlimits=dl
		d.v=fbins_64[0:63]
		store_data,tnames[i],data=d,limits=l,dlimits=dl
	endfor
endif


if bins eq 112 then begin

	tplot_names,'*efw_112_spec*',names=tnames
	
	
	; 112 bin spec
	fbins_112 = [findgen(32)*8,findgen(16)*16+256,$
			findgen(16)*32+512,findgen(16)*64+1024,$
			findgen(16)*128+2048,findgen(16)*256+4096]
	
	fcenter_112=(fbins_112[0:111] + fbins_112[1:112])/2.
	
	fbin_labels_112=strarr(112)
	fbin_labels_112[0:111]=string(fbins_112[0:111], format='(I0)')+'-'+$
				string(fbins_112[1:112], format='(I0)')+' Hz'
	
	
	for i=0,size(tnames,/n_elements)-1 do begin
		get_data,tnames[i],data=d,limits=l,dlimits=dl
		d.v=fbins_112[0:111]
		store_data,tnames[i],data=d,limits=l,dlimits=dl
	endfor
endif

end

