 
PRO neutral_sheet, time, pos, kp = kp, model = model, mlt = mlt, in_coord = incoord, $
                   distance2NS = distance2NS, z2ns=z2ns
              
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
; neutral_sheet - this routine will calculate the distance
;                 to the neutral sheet (d2NS). 
;                 Models include 'sm','themis', 'aen', 'den', 'fairfield',  
;                 'den-fairfield', 'lopez'. Default is 'themis'. More
;                 detailed information on each model can be found in 
;                 neutral_sheet_models.pro.
;
; *** WARNING *** These models have been initially verified but require 
;                 more testing. Use with caution!!!
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; 
; Inputs:
;
;   time - array of time stamps for position data. Time is double precision
;          number of seconds since Jan 01, 1970.
;   gsm_pos - position data, units are in RE, default coordinate system is gsm
;             pos=[x,y,z] (ex: x=pos[*,0], y=pos[*,1], z=pos[*,2]) an Nx3 array
;
; Input Keywords [optional]:
;   
;   model - name of the neutral sheet model used to generate distance data
;           Models include 'sm','themis', 'aen', 'den', 'fairfield', 'den-fairfield',
;           'lopez'. Default is 'themis'.
;   kp - kp value used by the lopez model. Default value is 0.
;   mlt - magnetic latitude in degrees used by the lopez model. Default is 0.0
;   in_coord - set this keyword to the input coordinate system if the data
;              is not in gsm coordiantes. Valid coordinate systems are: 
;              [gei, gsm, sm, gse, geo]
;
; Output Keywords:
;    
;   distance2NS - distance of spacecraft along the z axis to the neutral 
;                 sheet (in RE).
;
; Example:
;   neutral_sheet, time, gsm_pos, model='themis', distance2NS=d2NS
;
;  Modification History:
;    Initial Release - clrussell, 03-26-12
;
;  Notes:
;  1. The THM model returns the closest results at larger distances and 
;     the LM model - for smaller distances.
;  2. While the model scripts work, there is no such thing as the best 
;     or most accurate model.
;  3. 
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  ; validate and initialize parameters if not set
  IF ~keyword_set(model) THEN model = 'themis' ELSE model = strlowcase(model)
  models = ['sm', 'themis', 'aen', 'den', 'fairfield', 'den_fairfield', 'lopez']
  res = where(model EQ models, ncnt)
  IF ncnt LE 0 THEN BEGIN 
     print, 'An invalid neutral sheet model name was used. Valid entries include: '
     print, models 
     RETURN
  ENDIF

  ; source code for neutral sheet models
  neutral_sheet_models
  
  ; check input coordinate system, convert to gsm if needed
  IF ~keyword_set(in_coord) THEN in_coord = 'gsm' ELSE in_coord = strlowcase(in_coord)
  CASE in_coord OF
     'gsm': gsm_pos = pos
     'sm':  cotrans,pos,gsm_pos,time,/SM2GSM
     'gei': BEGIN
         cotrans,pos,gse_pos,time,/GEI2GSE
         cotrans,gse_pos,gsm_pos,time,/GSE2GSM
     END
     'gse': cotrans,pos,gsm_pos,time,/GSE2GSM
     'geo': BEGIN
         cotrans,pos,gei_pos,time,/GEO2GEI
         cotrans,gei_pos,gse_pos,time,/GEI2GSE     
         cotrans,gse_pos,gsm_pos,time,/GSE2GSM
     END
     ELSE: BEGIN
        print, 'Invalid coordinate system.' 
        print, 'Valid coordinate systems are: [gei, gsm, sm, gse, geo]'
        RETURN
     ENDELSE
  ENDCASE  
 
  ; call the appropriate neutral sheet model  
  CASE model OF
    'sm': distance2NS = sm_ns_model(time, gsm_pos, z2ns=z2ns)
    'themis': distance2NS = themis_ns_model(time, gsm_pos, z2ns=z2ns)
    'aen': distance2NS = aen_ns_model(time, gsm_pos, z2ns=z2ns)
    'den': distance2NS = den_ns_model(time, gsm_pos, z2ns=z2ns)
    'fairfield': distance2NS = fairfield_ns_model(time, gsm_pos, z2ns=z2ns)
    'den_fairfield': distance2NS = den_fairfield_ns_model(time, gsm_pos, z2ns=z2ns)
    'lopez': distance2NS = lopez_ns_model(time, gsm_pos, kp=kp, mlt=mlt, z2ns=z2ns)
    ELSE: BEGIN
        print, 'Invalid neutral sheet model.' 
        print, 'Valid models are: [sm, themis, aen, den, fairfield, den_fairfield, lopez]'
        RETURN
    ENDELSE
  ENDCASE

END
