
;+
;Procedure:
;  thm_part_slice2d_getticks
;
;
;Purpose:
;  Helper function for thm_part_slice2d_plot. 
;  Return an array of formatted annotation strings to be passed
;  to an IDL plotting procedure through the [xyz]tickname keyword.
;    
;
;Input:
;  nticks: (int) # of ticks requested by user (optional)
;  range: (float) two element array specifying axis range
;  log: (bool/int) flag to denote logarithmic axis, this should always
;                  be set to avoid persistence of the last axis setting
;  precision: (int) number of significant digits for annotation
;  style: (int) type of numberical annotation (0=auto, 1=decimal, 2=sci)
;  
;
;Output:
;  tickname: (string) Array of tick names
;  tickv: (float) Array of tick values
;  ticks: (int) Number of ticks - 1   
;
;
;Notes:
;  - This function should be called after the plot window has been initialized;
;    otherwise, the AXIS procedure will create an extra window.   
;  - If the # of ticks is not specified it will be determined by IDL.
;  - Associated tick values are returned via keyword (2013-April)
;
;
;$LastChangedBy: aaflores $
;$LastChangedDate: 2016-03-04 18:05:22 -0800 (Fri, 04 Mar 2016) $
;$LastChangedRevision: 20331 $
;$URL: svn+ssh://thmsvn@ambrosia.ssl.berkeley.edu/repos/spdsoft/trunk/projects/themis/spacecraft/particles/slices/plotting/thm_part_slice2d_getticks.pro $
;
;-
pro thm_part_slice2d_getticks, nticks=nticks, range=range, log=log, $
                                precision=precision, style=style, $
                                tickv=tickvals, tickname=ticknames, ticks=ticknum

    compile_opt idl2, hidden


  ;Adjust if # specified by user
  if size(nticks,/type) ne 0 then begin
    ticks = round(nticks) > 0 < 59
    ticks = ticks gt 1 ? ticks-1:(ticks > 1) ;n+1 plotted, =1 to supress
  endif
  
  ;Get the values of the default ticks generated by IDL 
  axis, yaxis=1, ystyle = 1+4, $
        yrange = range, ylog = log, $ 
        yticks = ticks, $
        ytick_get = tickvals

  ;Initialize output
  ticknames = strarr(n_elements(tickvals))
  ticknum = n_elements(tickvals)-1 

  ;Format ticks with custom routine
  for i=0, n_elements(tickvals)-1 do begin
    ticknames[i] = formatannotation(0,0,tickvals[i], $
      data={timeaxis:0,formatid:precision,scaling:0,exponent:style})
  endfor
  
  ;Use spaces to supress text for 0 or 1 ticks
  if size(nticks,/type) ne 0 then begin
    if nticks eq 1 then ticknames[1] = ' '
    if nticks lt 1 then ticknames[*] = ' '
  endif


end
