;+
;Procedure:
;  moka_mms_pad_plot
;
;Purpose:
;  To plot data generated by moka_mms_pad_fpi
;
;USAGE:
;  See 'moka_mms_pad_fpi_crib'
;  
;History:
;  2017-10-17 Created by Mitsuo Oka
;
;$LastChangedBy: moka $
;$LastChangedDate: 2017-09-30 11:03:14 -0700 (Sat, 30 Sep 2017) $
;$LastChangedRevision: 24073 $
;$URL: svn+ssh://ambrosia.ssl.berkeley.edu/repos/spdsoft/trunk/projects/mms/particles/moka/moka_mms_pad.pro $
;-
FUNCTION moka_mms_pad_plot_ydata, output, pad
  compile_opt idl2
  if n_elements(output) ne 1 then message,'output must be a scalar'
  tn = tag_names(pad)
  idx_omni = where(strmatch(tn,'SPEC_OMN'),ct_omni)
  idx_anti = where(strmatch(tn,'SPEC_180'),ct_anti)
  idx_perp = where(strmatch(tn,'SPEC__90'),ct_perp)
  idx_para = where(strmatch(tn,'SPEC___0'),ct_para)
  case output of
    'omni':     ydata =  ct_omni ? pad.SPEC_OMN : !VALUES.F_NAN 
    'anti-para':ydata =  ct_anti ? pad.SPEC_180 : !VALUES.F_NAN
    'perp':     ydata =  ct_perp ? pad.SPEC__90 : !VALUES.F_NAN
    'para':     ydata =  ct_para ? pad.SPEC___0 : !VALUES.F_NAN
    else:stop
  endcase
  return, ydata
END

PRO moka_mms_pad_plot, pad, second_pad, $
  ; Basic options
  output = output,  colors =colors, psym=psym, lblpos_x=lblpos_x, lblpos_y=lblpos_y,$
  second_output = second_output, second_colors=second_colors,second_psym=second_psym, $
  second_lblpos_x, second_lblpos_y, $
  ; Plot options
  xrange=xrange, $
  yrange=yrange, $
  plotsize=plotsize,$
  window=window, $
  noerase=noerase, $
  xunit = xunit, $ ; 'keV' or 'eV' or 'km/s'
  title = title, $
  nolabel = nolabel, $
  ; Additional plots
  add1=add1, add2=add2, add3=add3, $
  ; Eport
  export=export, eps=eps, $
  _extra=_extra
  
  compile_opt idl2
  
  lblpos_dy = 0.05
  if undefined(output) then output = 'omni'
  if undefined(colors) then colors = [0, 2, 4, 6, 1, 3, 5, 7]
  if undefined(psym) then psym = lonarr(8)-6L
  if undefined(lblpos_x) then lblpos_x = fltarr(8) + 0.8
  if undefined(lblpos_y) then lblpos_y = -findgen(8)*lblpos_dy + 0.95 
  if undefined(second_colors) then second_colors = colors
  if undefined(second_output) then second_output = output
  if undefined(second_psym) then second_psym = psym
  if undefined(noerase) then noerase = 0
  if undefined(xunit) then xunit = 'keV'
  case xunit of
    'km/s': xtitle_default = 'velocity [km/s]'
    'eV': xtitle_default = 'energy [eV]'
    'keV': xtitle_default = 'energy [keV]'
    else:message, 'Wrong xunit. Must of km/s, eV or keV'
  endcase
  if undefined(xtitle) then xtitle = xtitle_default
  if undefined(ytitle) then ytitle = pad.UNITS
  if undefined(title) then begin
    strtr = time_string(pad.TRANGE,prec=4)
    stime = strmid(strtr[0],11,100)
    etime = strmid(strtr[1],11,100)
    title = stime+' - '+etime
  endif
  cegy = strmatch(xunit,'keV') ? 0.001: 1. 
  xstyle=1
  ystyle=1
  if ~keyword_set(plotsize) then plotsize = 500.
  plotsize = 100. > plotsize ;min size
  
  ;---------------------------
  ; Get the (rough) dimensions of the plot
  ; The values here were largely determined by trial and error
  ;---------------------------
  plotxcsize = plotsize / !d.x_ch_size
  plotycsize = plotsize / !d.y_ch_size

  zprecision = 2
  xmargin = [11,15 + (zprecision-3 > 0)]
  ymargin = [4,2]

  tsize = strlen(title) * 1.25 ;title always 1.25 larger than other text

  xcsize = (total(xmargin) + plotxcsize) > tsize
  ycsize = (total(ymargin) + plotycsize)

  xsize = xcsize * !d.x_ch_size
  ysize = ycsize * !d.y_ch_size
  
  ;---------------------------
  ;Open postscript file
  ;---------------------------
  ; -specify aspect ratio to keep output on non-windows systems
  ;  from appearing clipped in that system's viewer
  if keyword_set(export) && keyword_set(eps) then begin
    popen, export, encapsulated = 1, $
      aspect = ysize/xsize, $
      _extra=_extra
  endif


  ;---------------------------
  ;Format the plotting window
  ;---------------------------
  if noerase eq 0 then begin
  if !d.name eq 'WIN' or !d.name eq 'X' and ~keyword_set(custom) then begin

    device, window_state = wins

    ;ensure a free window is used if possible
    if undefined(window) then begin
      wn = !d.window le 1 or !d.window ge 32  ?  $
        min( where(wins eq 0) ) > 2 : !d.window
    endif else begin
      wn = window
    endelse

    window, wn, xsize = xcsize * !d.x_ch_size, $
      ysize = ycsize * !d.y_ch_size, $
      title = title

    xmargin[0] = xmargin[0] > 0.5*(xcsize - plotxcsize)
  endif
  endif
  
  ;---------------------------
  ; Check Input
  ;---------------------------
  fail = 'The size of colors array need to be equal to (or larger than) that of output'
  imax1 = n_elements(output)
  if imax1 gt n_elements(colors) then message,fail
  imax2 = n_elements(second_output)
  if imax2 gt n_elements(second_colors) then message, fail
  
  ;---------------------------
  ; Plot Options
  ;---------------------------
  if ~undefined(second_pad) then begin
    if undefined(xrange) then begin
      wegy  = cegy*pad.EGY
      wegy2 = cegy*second_pad.EGY
      xrange = minmax([wegy,wegy2])
      xstyle = 0
    endif
    if undefined(yrange) then begin
      ydata1 = moka_mms_pad_plot_ydata(output[0],pad)
      ydata2 = moka_mms_pad_plot_ydata(output[0],second_pad)
      idx1 = where(ydata1 eq 0.,ct1)
      idx2 = where(ydata2 eq 0.,ct2)
      ydata1[idx1] = !VALUES.F_NAN
      ydata2[idx2] = !VALUES.F_NAN
      yrange = minmax([ydata1, ydata2])
      yrange[1] *= 10.
      ystyle = 0
    endif
  endif else begin
    if undefined(yrange) then begin
      ydata1 = moka_mms_pad_plot_ydata(output[0],pad)
      yrange = minmax(ydata1)
      yrange[1]*= 10.
    endif
  endelse
  
  xlog = ~strmatch(xunit,'km/s')
  
  if strmatch(xunit,'km/s') then begin
    case pad.species of
      'i': mass = !const.MP
      'e': mass = !const.ME
      else:message,'This species is not supported.'
    endcase
    mc2 = (mass*!const.c^2)/!const.E; kg*(m/s)^2 = J --> eV
    xdata0 = 0.001d0*!const.c*sqrt(1.d0 - 1.d0/((pad.EGY/mc2 + 1.d0)^2) )
  endif else begin
    xdata0 = cegy*pad.EGY
  endelse

  ;---------------------------
  ; PLOT first pad
  ;---------------------------
  i = 0
  xdata = (output[i] eq 'anti-para' and strmatch(xunit,'km/s')) ? (-1.)*xdata0 : xdata0 
  ydata = moka_mms_pad_plot_ydata(output[i], pad)
  plot, xdata, ydata, color=colors[i], title=title, $
    psym=psym[i], noerase=noerase, $
    xlog=xlog,xstyle=xstyle,xtitle=xtitle,xrange=xrange,$
    /ylog,ystyle=ystyle,ytitle=ytitle,yrange=yrange
  if ~undefined(xrange) and ~undefined(yrange) then begin
    if strmatch(xunit,'km/s') then begin
      xlblpos = xrange[0]+lblpos_x[i]*(xrange[1] - xrange[0])
    endif else begin
      xlblpos = 10^(alog10(abs(xrange[0]))+lblpos_x[i]*(alog10(abs(xrange[1]))-alog10(abs(xrange[0]))))
    endelse
    ylblpos = 10^(alog10(yrange[0])+lblpos_y[i]*(alog10(yrange[1])-alog10(yrange[0])))
    if ~keyword_set(nolabel) then xyouts,xlblpos, ylblpos, output[i],color=colors[i]
  endif
  if imax1 gt 1 then begin
    for i=1,imax1-1 do begin
      xdata = (output[i] eq 'anti-para' and strmatch(xunit,'km/s')) ? (-1.)*xdata0 : xdata0
      ydata = moka_mms_pad_plot_ydata(output[i],pad)
      if n_elements(ydata) gt 1 then begin
        oplot, xdata, ydata, color=colors[i], psym=psym[i]
        if ~undefined(xrange) and ~undefined(yrange) then begin
          if strmatch(xunit,'km/s') then begin
            xlblpos = xrange[0]+lblpos_x[i]*(xrange[1] - xrange[0])
          endif else begin
            xlblpos = 10^(alog10(abs(xrange[0]))+lblpos_x[i]*(alog10(abs(xrange[1]))-alog10(abs(xrange[0]))))
          endelse
          ylblpos = 10^(alog10(yrange[0])+lblpos_y[i]*(alog10(yrange[1])-alog10(yrange[0])))
          if ~keyword_set(nolabel) then xyouts,xlblpos, ylblpos, output[i],color=colors[i]
        endif
      endif
    endfor
  endif
  if strmatch(xunit,'km/s') then begin
    oplot,[0,0],yrange,linestyle=0,color=0
  endif
  ;---------------------------
  ; PLOT second pad
  ;---------------------------
  if ~undefined(second_pad) then begin
    if imax2 ge 1 then begin
      for i=0,imax2-1 do begin
        ydata = moka_mms_pad_plot_ydata(second_output[i],second_pad)
        if n_elements(ydata) gt 1 then begin
          oplot, cegy*second_pad.EGY, ydata, color=second_colors[i], psym=second_psym[i]
        endif
      endfor
    endif
  endif
  ;---------------------------
  ; Additional Plots
  ;---------------------------
  if n_tags(add1) ne 0 then begin
    oplot, cegy*add1.x, add1.y, linestyle=3, color=0
  endif
  if n_tags(add2) ne 0 then begin
    oplot, cegy*add2.x, add2.y, linestyle=3, color=0
  endif
  if n_tags(add3) ne 0 then begin
    oplot, cegy*add3.x, add3.y, linestyle=3, color=0
  endif

  ;----------------------------------------------
  ; Finish export:  write .png or close postscript file
  ;----------------------------------------------
  if keyword_set(export) then begin
    if keyword_set(eps) then begin
      pclose
    endif else begin
      makepng, export, /mkdir, _extra=_extra
    endelse
  endif
END